import json
from invirt.common import *
from os.path import getmtime

default_src_path   = '/etc/invirt/master.yaml'
default_cache_path = '/var/lib/invirt/cache.json'
lock_file          = '/var/lib/invirt/cache.lock'

def load(src_path = default_src_path,
         cache_path = default_cache_path,
         force_refresh = False):
    """
    Try loading the configuration from the faster-to-load JSON cache at
    cache_path.  If it doesn't exist or is outdated, load the configuration
    instead from the original YAML file at src_path and regenerate the cache.
    I assume I have the permissions to write to the cache directory.
    """

    # Namespace container for various state variables, so that they can be
    # updated by closures.
    ns = struct()

    if force_refresh:
        do_refresh = True
    else:
        src_mtime = getmtime(src_path)
        try:            cache_mtime   = getmtime(cache_path)
        except OSError: do_refresh = True
        else:           do_refresh = src_mtime > cache_mtime

    if not do_refresh:
        # Try reading from the cache first.  This must be transactionally
        # isolated from concurrent writes to prevent reading an incomplete
        # (changing) version of the data (but the transaction can share the
        # lock with other concurrent reads).  This isolation is accomplished
        # using an atomic filesystem rename in the refreshing stage.
        try: ns.cfg = with_closing(file(cache_path)) (
                lambda f: json.read(f.read()))
        except: do_refresh = True

    if do_refresh:
        # Atomically reload the source and regenerate the cache.  The read and
        # write must be a single transaction, or a stale version may be
        # written (if another read/write of a more recent configuration
        # is interleaved).  The final atomic rename is to keep this
        # transactionally isolated from the above cache read.  If we fail to
        # acquire the lock, just try to load the master configuration.
        import yaml
        try:    loader = yaml.CSafeLoader
        except: loader = yaml.SafeLoader
        try:
            @with_lock_file(lock_file)
            def refresh_cache():
                ns.cfg = with_closing(file(src_path)) (
                        lambda f: yaml.load(f, loader))
                try: with_closing(file(cache_path + '.tmp', 'w')) (
                        lambda f: f.write(json.write(ns.cfg)))
                except: pass # silent failure
                else: os.rename(cache_path + '.tmp', cache_path)
        except IOError:
            ns.cfg = with_closing(file(src_path)) (
                    lambda f: yaml.load(f, loader))
    return ns.cfg

dicts = load()
structs = dicts2struct(dicts)

# vim:et:sw=4:ts=4
