#!/usr/bin/python
from twisted.internet import reactor
from twisted.names import server
from twisted.names import dns
from twisted.names import common
from twisted.internet import defer
from twisted.python import failure

import sipb_xen_database
import psycopg2
import sqlalchemy
import time

class DatabaseAuthority(common.ResolverBase):
    """An Authority that is loaded from a file."""

    soa = None

    def __init__(self, domains, database=None):
        common.ResolverBase.__init__(self)
        if database is not None:
            sipb_xen_database.connect(database)
        self.domains = domains
        self.soa = dns.Record_SOA(mname='sipb-xen-dev.mit.edu', 
                                  rname='sipb-xen.mit.edu',
                                  serial=1, refresh=3600, retry=900,
                                  expire=3600000, minimum=21600, ttl=3600)
        self.ns = dns.Record_NS(name='ns1.xvm.mit.edu', ttl=3600)
        record = dns.Record_A(address='18.181.0.62', ttl=3600)
        self.ns1 = dns.RRHeader('ns1.xvm.mit.edu', dns.A, dns.IN,
                                3600, record, auth=True)

    
    def _lookup(self, name, cls, type, timeout = None):
        for i in range(3):
            try:
                value = self._lookup_unsafe(name, cls, type, timeout = None)
            except (psycopg2.OperationalError, sqlalchemy.exceptions.SQLError):
                if i == 2:
                    raise
                print "Reloading database"
                time.sleep(0.5)
                continue
            else:
                return value

    def _lookup_unsafe(self, name, cls, type, timeout):
        sipb_xen_database.clear_cache()
        
        ttl = 900
        if name.lower() in self.domains:
            domain = name.lower()
        else:
            # This works because domain will remain bound after breaking out of the loop
            for domain in self.domains:
                if name.lower().endswith('.'+domain):
                    break
            else: #Not us
                return defer.fail(failure.Failure(dns.DomainError(name)))
        results = []
        authority = []
        additional = [self.ns1]
        authority.append(dns.RRHeader(domain, dns.NS, dns.IN,
                                      3600, self.ns, auth=True))
        if cls == dns.IN:
            if type in (dns.A, dns.ALL_RECORDS):
                host = name[:-len(domain)-1]
                if not host:
                    record = dns.Record_A('18.181.0.62', ttl)
                    results.append(dns.RRHeader(name, dns.A, dns.IN, 
                                                ttl, record, auth=True))
                else:
                    value = sipb_xen_database.Machine.get_by(name=host)
                    if value is None or not value.nics:
                        return defer.fail(failure.Failure(dns.AuthoritativeDomainError(name)))
                    ip = value.nics[0].ip
                    if ip is None:  #Deactivated?
                        return defer.fail(failure.Failure(dns.AuthoritativeDomainError(name)))
                    record = dns.Record_A(ip, ttl)
                    results.append(dns.RRHeader(name, dns.A, dns.IN, 
                                                ttl, record, auth=True))
            elif type == dns.NS:
                results.append(dns.RRHeader(domain, dns.NS, dns.IN,
                                            ttl, self.ns, auth=True))
                authority = []
            elif type == dns.SOA:
                results.append(dns.RRHeader(domain, dns.SOA, dns.IN,
                                            ttl, self.soa, auth=True))
            return defer.succeed((results, authority, additional))
        else:
            #Doesn't exist
            return defer.fail(failure.Failure(dns.AuthoritativeDomainError(name)))

if '__main__' == __name__:
    resolver = DatabaseAuthority(['servers.csail.mit.edu',
                                  'xvm.mit.edu'],
                                 'postgres://sipb-xen@sipb-xen-dev/sipb_xen')

    verbosity = 0
    f = server.DNSServerFactory(authorities=[resolver], verbose=verbosity)
    p = dns.DNSDatagramProtocol(f)
    f.noisy = p.noisy = verbosity
    
    reactor.listenUDP(53, p)
    reactor.listenTCP(53, f)
    reactor.run()
