from __future__ import with_statement

import json
from invirt.common import *
import os
from os import rename
from os.path import getmtime
from contextlib import closing
import yaml
import re

try:    loader = yaml.CSafeLoader
except: loader = yaml.SafeLoader

src_path    = '/etc/invirt/master.yaml'
src_dirpath = '/etc/invirt/conf.d'
cache_path  = '/var/lib/invirt/cache.json'
lock_path   = '/var/lib/invirt/cache.lock'

def augment(d1, d2):
    """Splice dict-tree d2 into d1.  Return d1.

    d2 may be None for an empty dict-tree, because yaml.load produces that.

    Example:
    >>> d = {'a': {'b': 1}, 'c': 2}
    >>> augment(d, {'a': {'d': 3}})
    {'a': {'b', 1, 'd': 3}, 'c': 2}
    >>> d
    {'a': {'b', 1, 'd': 3}, 'c': 2}
    """
    if d2 is None:
        return d1
    for k in d2:
        if k in d1 and isinstance(d1[k], dict):
            augment(d1[k], d2[k])
        else:
            d1[k] = d2[k]
    return d1

def run_parts_list(dirname):
    """Reimplements Debian's run-parts --list.

    One difference from run-parts's behavior: run-parts --list /foo/
    will give output like /foo//bar, but run_parts_list('/foo/') gives
    /foo/bar in deference to Python conventions.

    Matches documented behavior of run-parts in debianutils v2.28.2, dated 2007.
    """
    # From run-parts(8).
    lanana_re   = re.compile('^[a-z0-9]+$')
    lsb_re      = re.compile('^_?([a-z0-9_.]+-)+[a-z0-9]+$')
    deb_cron_re = re.compile('^[a-z0-9][a-z0-9-]*$')
    for name in os.listdir(dirname):
        if lanana_re.match(name) or lsb_re.match(name) or deb_cron_re.match(name):
            yield os.path.join(dirname, name)

def list_files():
    yield src_path
    for name in run_parts_list(src_dirpath):
        yield name

def load_master():
    config = dict()
    for filename in list_files():
        with closing(file(filename)) as f:
            augment(config, yaml.load(f, loader))
    return config

def get_src_mtime():
    return max(max(getmtime(filename) for filename in list_files()),
               getmtime(src_dirpath))

def load(force_refresh = False):
    """
    Try loading the configuration from the faster-to-load JSON cache at
    cache_path.  If it doesn't exist or is outdated, load the configuration
    instead from the original YAML file at src_path and regenerate the cache.
    I assume I have the permissions to write to the cache directory.
    """

    # Namespace container for state variables, so that they can be updated by
    # closures.
    ns = struct()

    if force_refresh:
        do_refresh = True
    else:
        src_mtime = get_src_mtime()
        try:            cache_mtime = getmtime(cache_path)
        except OSError: do_refresh  = True
        else:           do_refresh  = src_mtime + 1 >= cache_mtime

        # We chose not to simply say
        #
        #   do_refresh = src_mtime >= cache_time
        #
        # because between the getmtime(src_path) and the time the cache is
        # rewritten, the master configuration may have been updated, so future
        # checks here would find a cache with a newer mtime than the master
        # (and thus treat the cache as containing the latest version of the
        # master).  The +1 means that for at least a full second following the
        # update to the master, this function will refresh the cache, giving us
        # 1 second to write the cache.  Note that if it takes longer than 1
        # second to write the cache, then this situation could still arise.
        #
        # The getmtime calls should logically be part of the same transaction
        # as the rest of this function (cache read + conditional cache
        # refresh), but to wrap everything in an flock would cause the
        # following cache read to be less streamlined.

    if not do_refresh:
        # Try reading from the cache first.  This must be transactionally
        # isolated from concurrent writes to prevent reading an incomplete
        # (changing) version of the data (but the transaction can share the
        # lock with other concurrent reads).  This isolation is accomplished
        # using an atomic filesystem rename in the refreshing stage.
        try: 
            with closing(file(cache_path)) as f:
                ns.cfg = json.read(f.read())
        except: do_refresh = True

    if do_refresh:
        # Atomically reload the source and regenerate the cache.  The read and
        # write must be a single transaction, or a stale version may be
        # written (if another read/write of a more recent configuration
        # is interleaved).  The final atomic rename is to keep this
        # transactionally isolated from the above cache read.  If we fail to
        # acquire the lock, just try to load the master configuration.
        try:
            with lock_file(lock_path):
                ns.cfg = load_master()
                try: 
                    with closing(file(cache_path + '.tmp', 'w')) as f:
                        f.write(json.write(ns.cfg))
                except: pass # silent failure
                else: rename(cache_path + '.tmp', cache_path)
        except IOError:
            ns.cfg = load_master()
    return ns.cfg

dicts = load()
structs = dicts2struct(dicts, '')
safestructs = dicts2struct(dicts, '', '')

# vim:et:sw=4:ts=4
