import validation
from invirt.common import CodeError, InvalidInput
import random
import sys
import time
import re
import cache_acls
import yaml

from invirt.config import structs as config
from invirt.database import Machine, Disk, Type, NIC, CDROM, session, meta
from invirt.remctl import remctl as gen_remctl

# ... and stolen from xend/uuid.py
def randomUUID():
    """Generate a random UUID."""

    return [ random.randint(0, 255) for _ in range(0, 16) ]

def uuidToString(u):
    """Turn a numeric UUID to a hyphen-seperated one."""
    return "-".join(["%02x" * 4, "%02x" * 2, "%02x" * 2, "%02x" * 2,
                     "%02x" * 6]) % tuple(u)
# end stolen code

def remctl(*args, **kwargs):
    return gen_remctl(config.remote.hostname,
                      principal='daemon/'+config.web.hostname,
                      *args, **kwargs)

def lvcreate(machine, disk):
    """Create a single disk for a machine"""
    remctl('web', 'lvcreate', machine.name,
           disk.guest_device_name, str(disk.size))
    
def makeDisks(machine):
    """Update the lvm partitions to add a disk."""
    for disk in machine.disks:
        lvcreate(machine, disk)

def getswap(disksize, memsize):
    """Returns the recommended swap partition size."""
    return int(min(disksize / 4, memsize * 1.5))

def lvinstall(machine, autoinstall):
    disksize = machine.disks[0].size
    memsize = machine.memory
    swapsize = getswap(disksize, memsize)
    imagesize = disksize - swapsize
    ip = machine.nics[0].ip
    remctl('control', machine.name, 'install', 
           'dist=%s' % autoinstall.distribution,
           'mirror=%s' % autoinstall.mirror,
           'arch=%s' % autoinstall.arch,
           'imagesize=%s' % imagesize)

def lvcopy(machine_orig_name, machine, rootpw):
    """Copy a golden image onto a machine's disk"""
    remctl('web', 'lvcopy', machine_orig_name, machine.name, rootpw)

def bootMachine(machine, cdtype):
    """Boot a machine with a given boot CD.

    If cdtype is None, give no boot cd.  Otherwise, it is the string
    id of the CD (e.g. 'gutsy_i386')
    """
    try:
        if cdtype is not None:
            out = remctl('control', machine.name, 'create', 
                              cdtype)
        else:
            out = remctl('control', machine.name, 'create')
    except CodeError, e:
        if 'already running' in e.message:
            raise InvalidInput('action', 'create',
                               'VM %s is already on' % machine.name)
        else:
        raise CodeError('"%s" on "control %s create %s' 
                        % (err, machine.name, cdtype))

def createVm(username, state, owner, contact, name, description, memory, disksize, machine_type, cdrom, autoinstall):
    """Create a VM and put it in the database"""
    # put stuff in the table
    session.begin()
    try:
        validation.Validate(username, state, name=name, description=description, owner=owner, memory=memory, disksize=disksize/1024.)
        machine = Machine()
        machine.name = name
        machine.description = description
        machine.memory = memory
        machine.owner = owner
        machine.administrator = None
        machine.contact = contact
        machine.uuid = uuidToString(randomUUID())
        machine.boot_off_cd = True
        machine.type = machine_type
        session.save_or_update(machine)
        disk = Disk(machine=machine,
                    guest_device_name='hda', size=disksize)
        nic = NIC.query().filter_by(machine_id=None).first()
        if not nic: #No IPs left!
            raise CodeError("No IP addresses left!  "
                            "Contact %s." % config.web.errormail)
        nic.machine = machine
        nic.hostname = name
        session.save_or_update(nic)
        session.save_or_update(disk)
        cache_acls.refreshMachine(machine)
        session.commit()
    except:
        session.rollback()
        raise
    makeDisks(machine)
    try:
        if autoinstall:
            lvinstall(machine, autoinstall)
        else:
            # tell it to boot with cdrom
            bootMachine(machine, cdrom)
    except CodeError, e:
        deleteVM(machine)
        raise
    return machine

def getList():
    """Return a dictionary mapping machine names to dicts."""
    value_string = remctl('web', 'listvms')
    value_dict = yaml.load(value_string, yaml.CSafeLoader)
    return value_dict

def parseStatus(s):
    """Parse a status string into nested tuples of strings.

    s = output of xm list --long <machine_name>
    """
    values = re.split('([()])', s)
    stack = [[]]
    for v in values[2:-2]: #remove initial and final '()'
        if not v:
            continue
        v = v.strip()
        if v == '(':
            stack.append([])
        elif v == ')':
            if len(stack[-1]) == 1:
                stack[-1].append('')
            stack[-2].append(stack[-1])
            stack.pop()
        else:
            if not v:
                continue
            stack[-1].extend(v.split())
    return stack[-1]

def statusInfo(machine):
    """Return the status list for a given machine.

    Gets and parses xm list --long
    """
    try:
        value_string = remctl('control', machine.name, 'list-long')
    except CodeError, e:
        if 'is not on' in e.message:
            return None
        else:
            raise
    status = parseStatus(value_string)
    return status

def listHost(machine):
    """Return the host a machine is running on"""
    try:
        out = remctl('control', machine.name, 'listhost')
    except CodeError, e:
        return None
    return out.strip()

def vnctoken(machine):
    """Return a time-stamped VNC token"""
    try:
        out = remctl('control', machine.name, 'vnctoken')
    except CodeError, e:
        return None
    return out.strip()

def deleteVM(machine):
    """Delete a VM."""
    try:
        remctl('control', machine.name, 'destroy')
    except CodeError:
        pass
    session.begin()
    delete_disk_pairs = [(machine.name, d.guest_device_name) 
                         for d in machine.disks]
    try:
        for mname, dname in delete_disk_pairs:
            remctl('web', 'lvremove', mname, dname)
        for nic in machine.nics:
            nic.machine_id = None
            nic.hostname = None
            session.save_or_update(nic)
        for disk in machine.disks:
            session.delete(disk)
        session.delete(machine)
        session.commit()
    except:
        session.rollback()
        raise

def commandResult(username, state, fields):
    start_time = 0
    machine = validation.Validate(username, state, machine_id=fields.getfirst('machine_id')).machine
    action = fields.getfirst('action')
    cdrom = fields.getfirst('cdrom')
    if cdrom is not None and not CDROM.query().filter_by(cdrom_id=cdrom).one():
        raise CodeError("Invalid cdrom type '%s'" % cdrom)    
    if action not in ('Reboot', 'Power on', 'Power off', 'Shutdown', 
                      'Delete VM'):
        raise CodeError("Invalid action '%s'" % action)
    if action == 'Reboot':
        try:
            if cdrom is not None:
                out = remctl('control', machine.name, 'reboot', cdrom)
            else:
                out = remctl('control', machine.name, 'reboot')
        except CodeError, e:
            if re.match("machine '.*' is not on", e.message):
                raise InvalidInput("action", "reboot", 
                                   "Machine is not on")
            else:
                raise
                
    elif action == 'Power on':
        if validation.maxMemory(username, state, machine) < machine.memory:
            raise InvalidInput('action', 'Power on',
                               "You don't have enough free RAM quota "
                               "to turn on this machine.")
        bootMachine(machine, cdrom)
    elif action == 'Power off':
        try:
            out = remctl('control', machine.name, 'destroy')
        except CodeError, e:
            if re.match("machine '.*' is not on", e.message):
                raise InvalidInput("action", "Power off", 
                                   "Machine is not on.")
            else:
                raise
    elif action == 'Shutdown':
        try:
            out = remctl('control', machine.name, 'shutdown')
        except CodeError, e:
            if re.match("machine '.*' is not on", e.message):
                raise InvalidInput("action", "Shutdown", 
                                   "Machine is not on.")
            else:
                raise
    elif action == 'Delete VM':
        deleteVM(machine)

    d = dict(user=username,
             command=action,
             machine=machine)
    return d

def resizeDisk(machine_name, disk_name, new_size):
    remctl("web", "lvresize", machine_name, disk_name, new_size)

def renameMachine(machine, old_name, new_name):
    for disk in machine.disks:
        remctl("web", "lvrename", old_name, 
               disk.guest_device_name, new_name)
    
