import unittest
from fcntl import flock, LOCK_EX, LOCK_UN
from os import remove

class struct(object):
    'A simple namespace object.'
    def __init__(self, d = {}, **kwargs):
        'd is the dictionary to update my __dict__ with.'
        self.__dict__.update(d)
        self.__dict__.update(kwargs)

def dicts2struct(x):
    """
    Given a tree of lists/dicts, perform a deep traversal to transform all the
    dicts to structs.
    """
    if type(x) == dict:
        return struct((k, dicts2struct(v)) for k,v in x.iteritems())
    elif type(x) == list:
        return [dicts2struct(v) for v in x]
    else:
        return x

#
# Hacks to work around lack of Python 2.5's `with` statement.
#

def with_closing(rsrc):
    "Utility to emulate Python 2.5's `with closing(rsrc)` context manager."
    def wrapper(func):
        try: return func(rsrc)
        finally: rsrc.close()
    return wrapper

def with_lock_file(path):
    """
    Context manager for lock files.  Example:

    @with_lock_file('/tmp/mylock')
    def input():
        print 'locked'
        return raw_input()
    # decorator is executed immediately
    print input # prints the input text
    """
    def wrapper(func):
        @with_closing(file(path, 'w'))
        def g(f):
            flock(f, LOCK_EX)
            try: return func()
            finally: flock(f, LOCK_UN)
        remove(path)
        return g
    return wrapper

#
# Tests.
#

class common_tests(unittest.TestCase):
    def test_dicts2structs(self):
        dicts = {
                'atom': 0,
                'dict': { 'atom': 'atom', 'list': [1,2,3] },
                'list': [ 'atom', {'key': 'value'} ]
                }
        structs = dicts2struct(dicts)
        self.assertEqual(structs.atom,        dicts['atom'])
        self.assertEqual(structs.dict.atom,   dicts['dict']['atom'])
        self.assertEqual(structs.dict.list,   dicts['dict']['list'])
        self.assertEqual(structs.list[0],     dicts['list'][0])
        self.assertEqual(structs.list[1].key, dicts['list'][1]['key'])

if __name__ == '__main__':
    unittest.main()
