"""
Functions to perform remctls.
"""

from sipb_xen_database import Machine, Disk, Type, NIC, CDROM, ctx, meta
import validation
from webcommon import CodeError, InvalidInput
import random
import subprocess
import sys
import time
import re
import cache_acls
import yaml

# ... and stolen from xend/uuid.py
def randomUUID():
    """Generate a random UUID."""

    return [ random.randint(0, 255) for _ in range(0, 16) ]

def uuidToString(u):
    """Turn a numeric UUID to a hyphen-seperated one."""
    return "-".join(["%02x" * 4, "%02x" * 2, "%02x" * 2, "%02x" * 2,
                     "%02x" * 6]) % tuple(u)
# end stolen code

def kinit(username = 'daemon/sipb-xen.mit.edu', keytab = '/etc/sipb-xen.keytab'):
    """Kinit with a given username and keytab"""

    p = subprocess.Popen(['kinit', "-k", "-t", keytab, username],
                         stderr=subprocess.PIPE)
    e = p.wait()
    if e:
        raise CodeError("Error %s in kinit: %s" % (e, p.stderr.read()))

def checkKinit():
    """If we lack tickets, kinit."""
    p = subprocess.Popen(['klist', '-s'])
    if p.wait():
        kinit()

def remctl(*args, **kws):
    """Perform a remctl and return the output.

    kinits if necessary, and outputs errors to stderr.
    """
    checkKinit()
    p = subprocess.Popen(['remctl', 'remote.mit.edu']
                         + list(args),
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE)
    v = p.wait()
    if kws.get('err'):
        return p.stdout.read(), p.stderr.read()
    if v:
        print >> sys.stderr, 'Error', v, 'on remctl', args, ':'
        print >> sys.stderr, p.stderr.read()
        raise CodeError('ERROR on remctl')
    return p.stdout.read()

def lvcreate(machine, disk):
    """Create a single disk for a machine"""
    remctl('web', 'lvcreate', machine.name,
           disk.guest_device_name, str(disk.size))
    
def makeDisks(machine):
    """Update the lvm partitions to add a disk."""
    for disk in machine.disks:
        lvcreate(machine, disk)

def lvcopy(machine_orig_name, machine, rootpw):
    """Copy a golden image onto a machine's disk"""
    remctl('web', 'lvcopy', machine_orig_name, machine.name, rootpw)

def bootMachine(machine, cdtype):
    """Boot a machine with a given boot CD.

    If cdtype is None, give no boot cd.  Otherwise, it is the string
    id of the CD (e.g. 'gutsy_i386')
    """
    if cdtype is not None:
        out, err = remctl('control', machine.name, 'create', 
                          cdtype, err=True)
    else:
        out, err = remctl('control', machine.name, 'create',
                          err=True)
    if 'already exists' in out:
        raise InvalidInput('action', 'create',
                           'VM %s is already on' % machine.name)
    elif err:
        raise CodeError('"%s" on "control %s create %s' 
                        % (err, machine.name, cdtype))

def createVm(username, state, owner, contact, name, memory, disksize, machine_type, cdrom, clone_from):
    """Create a VM and put it in the database"""
    # put stuff in the table
    transaction = ctx.current.create_transaction()
    try:
        validation.Validate(username, state, owner=owner, memory=memory, disksize=disksize/1024.)
        res = meta.engine.execute('select nextval('
                                  '\'"machines_machine_id_seq"\')')
        id = res.fetchone()[0]
        machine = Machine()
        machine.machine_id = id
        machine.name = name
        machine.memory = memory
        machine.owner = owner
        machine.administrator = owner
        machine.contact = contact
        machine.uuid = uuidToString(randomUUID())
        machine.boot_off_cd = True
        machine.type_id = machine_type.type_id
        ctx.current.save(machine)
        disk = Disk(machine_id=machine.machine_id,
                    guest_device_name='hda', size=disksize)
        open_nics = NIC.select_by(machine_id=None)
        if not open_nics: #No IPs left!
            raise CodeError("No IP addresses left!  "
                            "Contact xvm@mit.edu.")
        nic = open_nics[0]
        nic.machine_id = machine.machine_id
        nic.hostname = name
        ctx.current.save(nic)
        ctx.current.save(disk)
        cache_acls.refreshMachine(machine)
        transaction.commit()
    except:
        transaction.rollback()
        raise
    makeDisks(machine)
    if clone_from:
        lvcopy(clone_from, machine, 'password')
    # tell it to boot with cdrom
    bootMachine(machine, cdrom)
    return machine

def getList():
    """Return a dictionary mapping machine names to dicts."""
    value_string = remctl('web', 'listvms')
    value_dict = yaml.load(value_string, yaml.CSafeLoader)
    return value_dict

def parseStatus(s):
    """Parse a status string into nested tuples of strings.

    s = output of xm list --long <machine_name>
    """
    values = re.split('([()])', s)
    stack = [[]]
    for v in values[2:-2]: #remove initial and final '()'
        if not v:
            continue
        v = v.strip()
        if v == '(':
            stack.append([])
        elif v == ')':
            if len(stack[-1]) == 1:
                stack[-1].append('')
            stack[-2].append(stack[-1])
            stack.pop()
        else:
            if not v:
                continue
            stack[-1].extend(v.split())
    return stack[-1]

def statusInfo(machine):
    """Return the status list for a given machine.

    Gets and parses xm list --long
    """
    value_string, err_string = remctl('control', machine.name, 'list-long', 
                                      err=True)
    if 'Unknown command' in err_string:
        raise CodeError("ERROR in remctl list-long %s is not registered" % 
                        (machine.name,))
    elif 'does not exist' in err_string:
        return None
    elif err_string:
        raise CodeError("ERROR in remctl list-long %s:  %s" % 
                        (machine.name, err_string))
    status = parseStatus(value_string)
    return status

def deleteVM(machine):
    """Delete a VM."""
    remctl('control', machine.name, 'destroy', err=True)
    transaction = ctx.current.create_transaction()
    delete_disk_pairs = [(machine.name, d.guest_device_name) 
                         for d in machine.disks]
    try:
        for nic in machine.nics:
            nic.machine_id = None
            nic.hostname = None
            ctx.current.save(nic)
        for disk in machine.disks:
            ctx.current.delete(disk)
        for access in machine.acl:
            ctx.current.delete(access)
        ctx.current.delete(machine)
        transaction.commit()
    except:
        transaction.rollback()
        raise
    for mname, dname in delete_disk_pairs:
        remctl('web', 'lvremove', mname, dname)

def commandResult(username, state, fields):
    start_time = 0
    machine = validation.Validate(username, state, machine_id=fields.getfirst('machine_id')).machine
    action = fields.getfirst('action')
    cdrom = fields.getfirst('cdrom')
    if cdrom is not None and not CDROM.get(cdrom):
        raise CodeError("Invalid cdrom type '%s'" % cdrom)    
    if action not in ('Reboot', 'Power on', 'Power off', 'Shutdown', 
                      'Delete VM'):
        raise CodeError("Invalid action '%s'" % action)
    if action == 'Reboot':
        if cdrom is not None:
            out, err = remctl('control', machine.name, 'reboot', cdrom,
                              err=True)
        else:
            out, err = remctl('control', machine.name, 'reboot',
                              err=True)
        if err:
            if re.match("Error: Domain '.*' does not exist.", err):
                raise InvalidInput("action", "reboot", 
                                   "Machine is not on")
            else:
                print >> sys.stderr, 'Error on reboot:'
                print >> sys.stderr, err
                raise CodeError('ERROR on remctl')
                
    elif action == 'Power on':
        if validation.maxMemory(username, state, machine) < machine.memory:
            raise InvalidInput('action', 'Power on',
                               "You don't have enough free RAM quota "
                               "to turn on this machine.")
        bootMachine(machine, cdrom)
    elif action == 'Power off':
        out, err = remctl('control', machine.name, 'destroy', err=True)
        if err:
            if re.match("Error: Domain '.*' does not exist.", err):
                raise InvalidInput("action", "Power off", 
                                   "Machine is not on.")
            else:
                print >> sys.stderr, 'Error on power off:'
                print >> sys.stderr, err
                raise CodeError('ERROR on remctl')
    elif action == 'Shutdown':
        out, err = remctl('control', machine.name, 'shutdown', err=True)
        if err:
            if re.match("Error: Domain '.*' does not exist.", err):
                raise InvalidInput("action", "Shutdown", 
                                   "Machine is not on.")
            else:
                print >> sys.stderr, 'Error on Shutdown:'
                print >> sys.stderr, err
                raise CodeError('ERROR on remctl')
    elif action == 'Delete VM':
        deleteVM(machine)

    d = dict(user=username,
             command=action,
             machine=machine)
    return d

def resizeDisk(machine_name, disk_name, new_size):
    remctl("web", "lvresize", machine_name, disk_name, new_size)

def renameMachine(machine, old_name, new_name):
    for disk in machine.disks:
        remctl("web", "lvrename", old_name, 
               disk.guest_device_name, new_name)
    
