"""Exceptions for the web interface."""

from sipb_xen_database import Machine, MachineAccess

class MyException(Exception):
    """Base class for my exceptions"""
    pass

class InvalidInput(MyException):
    """Exception for user-provided input is invalid but maybe in good faith.

    This would include setting memory to negative (which might be a
    typo) but not setting an invalid boot CD (which requires bypassing
    the select box).
    """
    def __init__(self, err_field, err_value, expl=None):
        MyException.__init__(self, expl)
        self.err_field = err_field
        self.err_value = err_value

class CodeError(MyException):
    """Exception for internal errors or bad faith input."""
    pass

import controls

def cachedproperty(func):
    name = '__cache_' + func.__name__ + '_' + str(id(func))
    def getter(self):
        try:
            return getattr(self, name)
        except AttributeError:
            value = func(self)
            setattr(self, name, value)
            return value
    return property(getter)

class Global(object):
    """Global state of the system, to avoid duplicate remctls to get state"""
    def __init__(self, user):
        self.user = user
    
    machines = cachedproperty(lambda self: 
                             [ma.machine for ma in 
                              MachineAccess.select_by(user=self.user)])
    uptimes = cachedproperty(lambda self: 
                             controls.getUptimes(self.machines))

    def clear(self):
        """Clear the state so future accesses reload it."""
        for attr in list(self.__dict__):
            if attr.startswith('__cache_'):
                delattr(self, attr)

g = Global(None)
