from sqlalchemy import *

from sqlalchemy.ext.sessioncontext import SessionContext
from sqlalchemy.ext.assignmapper import assign_mapper

__all__ = ['meta',
           'ctx',
           'clear_cache',
           'machine_table',
           'machine_access_table',
           'nic_table',
           'disk_table',
           'types_table',
           'cdroms_table',
           'Machine',
           'MachineAccess',
           'NIC',
           'Disk',
           'Type',
           'CDROM']

meta = DynamicMetaData()
ctx = SessionContext(create_session)

machine_table = Table('machines', meta,
       Column('machine_id', Integer, primary_key=True, nullable=False),
       Column('name', String, nullable=False),
       Column('memory', Integer, nullable=False),
       Column('owner', String, nullable=False),
       Column('contact', String, nullable=False),
       Column('uuid', String, nullable=False),
       Column('administrator', String, nullable=False, default=False),
       Column('type_id', String, ForeignKey('types.type_id'), nullable=False),
       Column('autorestart', Boolean, nullable=False, default=False),
       Column('cpus', Integer, nullable=False, default=1))

nic_table = Table('nics', meta,
       Column('machine_id', Integer, ForeignKey('machines.machine_id'), nullable=True),
       Column('mac_addr', String, nullable=False, primary_key=True),
       Column('ip', String, nullable=False, unique=True),
       Column('hostname', String, nullable=True))

disk_table = Table('disks', meta,
       Column('machine_id', Integer, ForeignKey('machines.machine_id'), nullable=False),
       Column('guest_device_name', String, nullable=False),
       Column('size', Integer, nullable=False),
       PrimaryKeyConstraint('machine_id', 'guest_device_name'))

types_table = Table('types', meta,
       Column('type_id', String, primary_key=True, nullable=False),
       Column('description', String, nullable=False),
       Column('hvm', Boolean, nullable=False),
       Column('apic', Boolean, nullable=False),
       Column('acpi', Boolean, nullable=False),
       Column('pae', Boolean, nullable=False))

cdroms_table = Table('cdroms', meta,
       Column('cdrom_id', String, primary_key=True, nullable=False),
       Column('description', String, nullable=False))

machine_access_table = Table('machine_access', meta,
       Column('machine_id', Integer, ForeignKey('machines.machine_id'), nullable=False, index=True),
       Column('user', String, nullable=False, index=True),
       PrimaryKeyConstraint('machine_id', 'user'))

class Machine(object):
    def __repr__(self):
        return "<Machine %s: name='%s' owner='%s'>" % (self.machine_id, self.name, self.owner)

class MachineAccess(object):
    def __repr__(self):
        return "<MachineAccess machine='%s' user='%s'>" % (self.machine, self.user)

class NIC(object):
    def __repr__(self):
        return "<NIC: mac='%s' machine='%s' ip='%s' hostname='%s'>" % (self.mac_addr, self.machine_id, self.ip, self.hostname)

class Disk(object):
    def __repr__(self):
        return "<Disk: machine=%s device=%s size=%s>" % (self.machine_id, self.guest_device_name, self.size)

class Type(object):
    def __repr__(self):
        return "<Type %s: %s>" % (self.type_id, self.description)

class CDROM(object):
    def __repr__(self):
        return "<CDROM %s: %s>" % (self.cdrom_id, self.description)

assign_mapper(ctx, Machine, machine_table,
              properties={'nics': relation(NIC, backref="machine"),
                          'disks': relation(Disk, backref="machine"),
                          'type': relation(Type),
                          'acl': relation(MachineAccess, backref="machine")});
assign_mapper(ctx, MachineAccess, machine_access_table)
assign_mapper(ctx, NIC, nic_table)
assign_mapper(ctx, Disk, disk_table)
assign_mapper(ctx, Type, types_table)
assign_mapper(ctx, CDROM, cdroms_table)

def clear_cache():
    """Clear sqlalchemy's cache.

    This _seems_ to be the way; it works, but the docs don't mention
    it.  Why is this so obscure?"""

    ctx.registry.clear()
