from __future__ import with_statement

import unittest
from fcntl import flock, LOCK_EX, LOCK_SH, LOCK_UN
import contextlib as clib
import subprocess

class InvirtConfigError(AttributeError):
    pass

class struct(dict):
    'A simple namespace object.'
    def __init__(self, d = {}, __prefix = None, __default=None, **kwargs):
        super(struct, self).__init__(d)
        self.__prefix = __prefix
        self.__default = __default
        self.update(kwargs)
    def __getattr__(self, key):
        try:
            return self[key]
        except KeyError:
            if self.__default is None:
                # XX ideally these would point a frame higher on the stack.
                prefix = self.__prefix
                if prefix is not None:
                    raise InvirtConfigError('missing configuration variable '
                                            '%s%s' % (prefix, key))
                else:
                    raise AttributeError("anonymous struct has no member '%s'"
                                         % (key,))
            else:
                return struct({}, '', self.__default)

def dicts2struct(x, prefix = None, default = None):
    """
    Given a tree of lists/dicts, perform a deep traversal to transform all the
    dicts to structs.
    """
    if prefix is not None:
        def newprefix(k): return prefix + str(k) + '.'
    else:
        def newprefix(k): return prefix
    if type(x) == dict:
        return struct(((k, dicts2struct(v, newprefix(k), default))
                       for k,v in x.iteritems()),
                      prefix,
                      default)
    elif type(x) == list:
        return [dicts2struct(v, newprefix(i), default)
                for i, v in enumerate(x)]
    elif x is None:
        return struct({}, prefix, default)
    else:
        return x

@clib.contextmanager
def lock_file(path, exclusive = True):
    with clib.closing(file(path, 'w')) as f:
        if exclusive:
            locktype = LOCK_EX
        else:
            locktype = LOCK_SH
        flock(f, locktype)
        try:
            yield
        finally:
            flock(f, LOCK_UN)

def captureOutput(popen_args, stdin_str=None, *args, **kwargs):
    """Capture stdout from a command.

    This method will proxy the arguments to subprocess.Popen. It
    returns the output from the command if the call succeeded and
    raises an exception if the process returns a non-0 value.

    This is intended to be a variant on the subprocess.check_call
    function that also allows you access to the output from the
    command.
    """
    if 'stdin' not in kwargs:
        kwargs['stdin'] = subprocess.PIPE
    if 'stdout' not in kwargs:
        kwargs['stdout'] = subprocess.PIPE
    if 'stderr' not in kwargs:
        kwargs['stderr'] = subprocess.STDOUT
    p = subprocess.Popen(popen_args, *args, **kwargs)
    out, _ = p.communicate(stdin_str)
    if p.returncode:
        raise subprocess.CalledProcessError(p.returncode, popen_args, out)
    return out

#
# Exceptions.
#

class InvalidInput(Exception):
    """Exception for user-provided input is invalid but maybe in good faith.

    This would include setting memory to negative (which might be a
    typo) but not setting an invalid boot CD (which requires bypassing
    the select box).
    """
    def __init__(self, err_field, err_value, expl=None):
        Exception.__init__(self, expl)
        self.err_field = err_field
        self.err_value = err_value

class CodeError(Exception):
    """Exception for internal errors or bad faith input."""
    pass

#
# Tests.
#

class common_tests(unittest.TestCase):
    def test_dicts2structs(self):
        dicts = {
                'atom': 0,
                'dict': { 'atom': 'atom', 'list': [1,2,3] },
                'list': [ 'atom', {'key': 'value'} ]
                }
        structs = dicts2struct(dicts, '')
        self.assertEqual(structs.atom,        dicts['atom'])
        self.assertEqual(structs.dict.atom,   dicts['dict']['atom'])
        self.assertEqual(structs.dict.list,   dicts['dict']['list'])
        self.assertEqual(structs.list[0],     dicts['list'][0])
        self.assertEqual(structs.list[1].key, dicts['list'][1]['key'])
        self.assertEqual(set(structs), set(['atom', 'dict', 'list']))

if __name__ == '__main__':
    unittest.main()
