# Anemon Dhcp
# Copyright (C) 2005 Mathieu Ignacio -- mignacio@april.org
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA

from datetime import datetime, timedelta

from pydhcplib.dhcp_constants import *
from pydhcplib.type_ipv4 import ipv4
from pydhcplib.type_hw_addr import hwmac
from pydhcplib.type_strlist import strlist

from anemon.server.type_ipv4subnet import ipv4subnet
from anemon.server.type_ipv4range import ipv4range
from anemon.server.event_logger import Log
from anemon.server import db_classes
from anemon.server import db_functions


class DhcpCore:
    def __init__(self) :
        Log.Output(Log.debug,"anemon_db : __init__")
        self.default_giaddr = "0.0.0.0"


    """ Private searching methods """

    def SetDefaultGiaddr(self,giaddr) :
        self.default_giaddr = giaddr


    """ Public request methods """
    def GetFreeIp(self,packet) :
        """
        Search and retrieve next free ip 
        Probably the most complexe part of anemon
        """

        Log.Output(Log.debug,"dhcp_core : GetFreeIp from ")

        clientid = strlist(packet.GetClientIdentifier())
        chaddr = hwmac(packet.GetHardwareAddress())
        giaddr = ipv4(packet.GetGiaddr())



        if (not giaddr) or giaddr=="0.0.0.0": giaddr = ipv4(self.default_giaddr)


        # 1. Get a lot of data
        ALLHOST = db_functions.SelectSingleHostScopes()





        # 2. Search previous lease with same chaddr and identifier
        previous = db_functions.SelectLeaseByChaddrAndIdentifier(chaddr.str(),clientid.str())




        # 3. if previous lease found, check all single host definition to know
        # if this host is allways allowed to get the same ip.
        if (previous) and (len(previous)>0) :
            ip = ipv4(previous[0].ipaddr)
            hostICI = db_functions.SelectSingleHostByICI(ip,chaddr,clientid)
            if (hostICI) and (len(hostICI)>0) :
                Log.Output(Log.warn,"dhcp_core : GetFreeIp = ip found by previous lease ("+ip.str()+")")
                return [ip,ip,ip]

                
        
        # 4. Search Ip by client_identifier
        if (clientid) :
            ip = db_functions.SelectIpFromHostByIdentifier(clientid)
            if (ip) :
                Log.Output(Log.warn,"dhcp_core : GetFreeIp = ip found by identifier ("+clientid.str()+")")
                return [ip,ip,ip]


        # 5. Search Ip by hardware address (chaddr)
        if (chaddr) :
            ip = db_functions.SelectIpFromHostByChaddr(chaddr)
            if (ip) :
                Log.Output(Log.warn,"dhcp_core : GetFreeIp = ip found by chaddr ("+chaddr.str()+")")
                return [ip,ip,ip]

        # 6. Search next free Ip by gateway address (giaddr)

        ALLSCOPES = db_functions.SelectMultipleHostScopes()
        # Get all scopes in conf corresponding to giaddr
        GIADDRSCOPES = [];
        for each in ALLSCOPES :
            if ipv4subnet(each.subnet).IsIpInNetwork(giaddr) : GIADDRSCOPES.append(each)

        for scope in GIADDRSCOPES :
            # a. For each scope getnextip
            iprange = ipv4range(scope.first,scope.last)
            ip = iprange.GetNextIp()
            # fixme : check for single host declaration before
            while ip.int() :
                # b. if free return ip
                if  self.LeaseLockTest(ip,chaddr) :
                    Log.Output(Log.warn,"dhcp_core : GetFreeIp = ip found by giaddr ("+giaddr.str()+")")
                    return [ip,ipv4(scope.first),ipv4(scope.last)]
                # c. if not free goto b.
                ip = iprange.GetNextIp()
                
        Log.Output(Log.warn,"dhcp_core : GetFreeIp = no free ip found")


        return False

    def GetHostParameters(self,first,last):
        """Read host and class database to make individual
        configuration parameters, this function will probably use a cache later"""

        Log.Output(Log.debug,"dhcp_core : GetHostParameters ("+first.str()+" - "+last.str()+")")

        options = {}

        for sqloption in db_functions.GetOptionsFromRange(first,last) :
            option_type = DhcpOptionsTypes[DhcpOptions[sqloption.parameter]]

            if option_type == "ipv4" :
                # this is a single ip address
                options[sqloption.parameter] = map(int,sqloption.value.split("."))
            elif option_type == "ipv4+" :
                # this is multiple ip address
                iplist = sqloption.value.split(",")
                opt = []
                for single in iplist :
                    opt.append(ipv4(single).list())
                options[sqloption.parameter] = opt
            elif option_type == "32-bits" :
                # This is probably a number...
                digit = int(sqloption.value)
                options[sqloption.parameter] = [digit>>24&0xFF,(digit>>16)&0xFF,(digit>>8)&0xFF,digit&0xFF]
            elif option_type == "16-bits" :
                digit = int(sqloption.value)
                options[sqloption.parameter] = [(digit>>8)&0xFF,digit&0xFF]

            elif option_type == "char" :
                digit = int(sqloption.value)
                options[sqloption.parameter] = [digit&0xFF]

            elif option_type == "bool" :
                if sqloption.value=="False" or sqloption.value=="false" or sqloption.value==0 :
                    options[sqloption.parameter] = [0]
                else : options[sqloption.parameter] = [1]
                    
            elif option_type == "string" :
                options[sqloption.parameter] = strlist(sqloption.value).list()
                
            else :
                options[sqloption.parameter] = strlist(sqloption.value).list()
        
        return options


    """ Leases functions below """

    def LeaseLockTest(self,ip,chaddr=hwmac("00:00:00:00:00:00")) :
        """ Check if lease is in use (locked)
        return True if lease is not found, False else. """
        Log.Output(Log.debug,"dhcp_core : LeaseLockTest")

        
        db_functions.UpdateLeaseState()

        leases = list(db_classes.Leases.select(db_classes.Leases.q.ipaddr==ip.str()))

        #(a) Check if no lease is return 
        if len(leases) == 0 :
            Log.Output(Log.debug,"dhcp_core  : lease not locked")
            return 1

        #(b) if leases are return, check if the ip was previously allocated by same host
        for each in leases :
            if each.chaddr == chaddr.str() :
                Log.Output(Log.debug,"dhcp_core  : lease previously locked for this mac address, ok.")
                return 2

        #(c) else, lease is allready locked by someone else
        Log.Output(Log.debug,"dhcp_core  : lease allready locked")
        return False


    def LeaseLock(self,packet) :
        """Mark lease as used """
        Log.Output(Log.debug,"dhcp_core : LeaseLock")

        ip = ipv4(packet.GetOption("yiaddr"))
        chaddr = hwmac(packet.GetHardwareAddress())
        leasetime = packet.GetOption("ip_address_lease_time")

        if (leasetime == False) or (leasetime == []) or (len(leasetime)!=4):
            leasetime = 99999
        else :
            # ipv4 : trick to convert leasetime list to int
            leasetime = ipv4(leasetime).int()

        lockstatus = self.LeaseLockTest(ip,chaddr)
        if lockstatus == 1 :
            # Lease not locked
            db_functions.InsertLease("",ip.str(),chaddr.str(),"",leasetime)
            return True

        elif lockstatus == 2 :
            # Lease locked by same host
            db_functions.UpdateLeaseTime(ip.str(),chaddr.str(),"",leasetime)
            return True

        # lockstatus not 1 and not 2 : Lease locked by another host
        return False


    def LeaseUnlock(self,packet) :
        """ Parameters needed : ip address of client"""
        Log.Output(Log.debug,"dhcp_core : LeaseUnlock")
        ip = ipv4(packet.GetOption("yiaddr"))
        chaddr = hwmac(packet.GetHardwareAddress())
        db_functions.DeleteLease(ip.str(),chaddr.str(),"")
        return True

