from __future__ import with_statement

import unittest
from fcntl import flock, LOCK_EX, LOCK_SH, LOCK_UN
import contextlib as clib

class InvirtConfigError(AttributeError):
    pass

class struct(object):
    'A simple namespace object.'
    def __init__(self, d = {}, __prefix = None, **kwargs):
        'd is the dictionary or the items-iterable to update my __dict__ with.'
        self.__dict__.update(d)
        self.__dict__.update(kwargs)
        self.__prefix = __prefix
    def __getattr__(self, key):
        # XX ideally these would point a frame higher on the stack.
        prefix = self.__prefix
        if prefix is not None:
            raise InvirtConfigError('missing configuration variable %s%s'
                                    % (prefix, key))
        else:
            raise AttributeError("anonymous struct has no member '%s'"
                                 % (key,))

def dicts2struct(x, prefix = None):
    """
    Given a tree of lists/dicts, perform a deep traversal to transform all the
    dicts to structs.
    """
    if prefix is not None:
        def newprefix(k): return prefix + str(k) + '.'
    else:
        def newprefix(k): return prefix
    if type(x) == dict:
        return struct(((k, dicts2struct(v, newprefix(k)))
                       for k,v in x.iteritems()),
                      prefix)
    elif type(x) == list:
        return [dicts2struct(v, newprefix(i)) for i, v in enumerate(x)]
    elif x is None:
        return struct({}, prefix)
    else:
        return x

@clib.contextmanager
def lock_file(path, exclusive = True):
    with clib.closing(file(path, 'w')) as f:
        if exclusive:
            locktype = LOCK_EX
        else:
            locktype = LOCK_SH
        flock(f, locktype)
        try:
            yield
        finally:
            flock(f, LOCK_UN)

#
# Exceptions.
#

class InvalidInput(Exception):
    """Exception for user-provided input is invalid but maybe in good faith.

    This would include setting memory to negative (which might be a
    typo) but not setting an invalid boot CD (which requires bypassing
    the select box).
    """
    def __init__(self, err_field, err_value, expl=None):
        Exception.__init__(self, expl)
        self.err_field = err_field
        self.err_value = err_value

class CodeError(Exception):
    """Exception for internal errors or bad faith input."""
    pass

#
# Tests.
#

class common_tests(unittest.TestCase):
    def test_dicts2structs(self):
        dicts = {
                'atom': 0,
                'dict': { 'atom': 'atom', 'list': [1,2,3] },
                'list': [ 'atom', {'key': 'value'} ]
                }
        structs = dicts2struct(dicts, '')
        self.assertEqual(structs.atom,        dicts['atom'])
        self.assertEqual(structs.dict.atom,   dicts['dict']['atom'])
        self.assertEqual(structs.dict.list,   dicts['dict']['list'])
        self.assertEqual(structs.list[0],     dicts['list'][0])
        self.assertEqual(structs.list[1].key, dicts['list'][1]['key'])

if __name__ == '__main__':
    unittest.main()
